function h = bus(varargin)
% BUS — 3D plot of three STANDARD cells (no cuts), aligned along +Y,
% with two side filler cells (left & right), each sized 40x30 in XxY
% and Z equal to the total bus height. Filler has 4 vertical layers:
% [SiO2 (H+H_Ti), Au (H_gold_bottom), HfO2 (H_vac), SiO2 (H_Au_top)].
%
% Usage:
%   bus;                               % draw with defaults (3 standard cells + fillers)
%   bus('Origin',[0 0 0]);             % change origin
%   bus('Axes',axes);                  % draw on existing axes
%   bus('Pitch','auto');               % pitch = W (default)
%   bus('Pitch',12);                   % custom inter-cell pitch in nm
%
% Output:
%   h.fig, h.ax = figure and axes handles

% ---------- Default geometry (nm) ----------
P = struct( ...
    'L',24, 'W',10, 'H',2, ...                % SiO2 base
    'origin',[0 0 0], ...                     % [x0 y0 z0]
    'H_Ti_bottom',1, ...                      % Ti bottom (height only)
    'H_gold_bottom',2, ...                    % Au bottom (height only)
    'L_vac',4, 'H_vac',3, ...                 % trench (Lx, Hz)
    'L_Au_top',16, 'W_Au_top',8, 'H_Au_top',4, ... % top Au (Lx, Wy, Hz)
    'T_film',0.5 ...                          % Ti rim thickness
);

opt = struct( ...
    'Origin',P.origin, ...
    'Axes',[], 'Figure',[], ...
    'Pitch','auto', ...       % 'auto' -> W
    'View',[35 25], ...
    'Title','Bus: three STANDARD cells (no cuts) + side fillers');

% ---------- Parse name–value ----------
if mod(numel(varargin),2)~=0
    error('Arguments must be name-value pairs.');
end
for k=1:2:numel(varargin)
    name=lower(varargin{k}); val=varargin{k+1};
    switch name
        case 'origin',  opt.Origin = val;
        case 'axes',    opt.Axes   = val;
        case 'figure',  opt.Figure = val;
        case 'pitch',   opt.Pitch  = val;
        case 'view',    opt.View   = val;
        case 'title',   opt.Title  = val;
        otherwise, error('Unknown option: %s', varargin{k});
    end
end
P.origin = opt.Origin;

% ---------- Prepare figure/axes ----------
newFig = isempty(opt.Axes);
if newFig
    if isempty(opt.Figure) || ~isgraphics(opt.Figure,'figure')
        h.fig = figure('Color','w','Position',[80 80 960 720]);
    else
        h.fig = opt.Figure;
    end
    h.ax = axes('Parent',h.fig);
else
    h.ax = opt.Axes;
    h.fig = ancestor(h.ax,'figure');
end

% ---------- Draw 3 STANDARD cells + two filler cells ----------
N = 3; % exactly three standard cells
if ischar(opt.Pitch) && strcmpi(opt.Pitch,'auto')
    pitch = P.W;             % standard pitch = full cell width
else
    pitch = max(0, double(opt.Pitch));
    if pitch==0, pitch = P.W; end
end

drawBusWithFillers(h.ax, P, pitch, N, opt.View, opt.Title);

if nargout==0, clear h; end
end

% ======================================================================
% ========================= Drawing functions ==========================
% ======================================================================
function drawBusWithFillers(ax, P, pitch, N, viewAngles, ttl)
[clr_SiO2,clr_Ti,clr_Au,clr_vac,a_vac,clr_HfO2,edgeColor,lw] = palette();

if ~isgraphics(ax), ax=gca; end
cla(ax,'reset'); delete(findall(ax,'Type','legend')); hold(ax,'on');

x0 = P.origin(1); y0 = P.origin(2); z0 = P.origin(3);

% ----- draw the 3 STANDARD cells (no cuts) -----
legendDone = false;
for i=1:N
    o = [x0, y0 + (i-1)*pitch, z0];
    showLegend = ~legendDone;

    drawOneStandardCell(ax, P, o, showLegend, ...
        clr_SiO2,clr_Ti,clr_Au,clr_vac,a_vac,clr_HfO2,edgeColor,lw);

    if showLegend, legendDone = true; end
end

% ----- total height (Z) of the bus -----
H_tot = P.H + P.H_Ti_bottom + P.H_gold_bottom + P.H_vac + P.H_Au_top;

% ----- two side FILLER cells (40 x 30 x H_tot), 4 layers -----
LxFill = 40;        % X size
WyFill = 30;        % Y size
h1 = P.H + P.H_Ti_bottom;    % layer 1 -> SiO2 color
h2 = P.H_gold_bottom;        % layer 2 -> Au color
h3 = P.H_vac;                % layer 3 -> HfO2 color
h4 = P.H_Au_top;             % layer 4 -> SiO2 color
hvec = [h1 h2 h3 h4];

% place them flush to the left and right of the bus bounding box in X
oFillL = [x0 - LxFill, y0, z0];      % left of the bus
oFillR = [x0 + P.L,    y0, z0];      % right of the bus

FL = buildFillerLayers(oFillL, LxFill, WyFill, hvec);
FR = buildFillerLayers(oFillR, LxFill, WyFill, hvec);

F = faces6();
% draw left filler (legend entry once)
drawFillerLayers(ax, FL, clr_SiO2, clr_Au, clr_HfO2, clr_SiO2, edgeColor, lw, F, 'FILLER (layered)');
% draw right filler (no legend)
drawFillerLayers(ax, FR, clr_SiO2, clr_Au, clr_HfO2, clr_SiO2, edgeColor, lw, F, '', true);

% --- Axes, view, limits
axis(ax,'equal'); grid(ax,'on'); view(ax,viewAngles(1),viewAngles(2));
xlabel(ax,'X [nm]'); ylabel(ax,'Y [nm]'); zlabel(ax,'Z [nm]');
title(ax, sprintf('%s — pitch = %.3g nm — H_{tot}=%.3g nm', ttl, pitch, H_tot));

Y_span = (N-1)*pitch + P.W;
m = 0.10 .* [P.L + 2*LxFill, max([P.W,Y_span,WyFill]), H_tot];

xlim(ax, [x0 - LxFill - m(1), x0 + P.L + LxFill + m(1)]);
ylim(ax, [y0 - m(2),          y0 + max(Y_span,WyFill) + m(2)]);
zlim(ax, [z0 - m(3),          z0 + H_tot + m(3)]);

plot3(ax,x0,y0,z0,'ko','MarkerFaceColor','k','HandleVisibility','off');
legend(ax,'Location','bestoutside');
hold(ax,'off'); drawnow limitrate;
end

function drawOneStandardCell(ax, P, o, showLegend, ...
        c_SiO2,c_Ti,c_Au,c_vac,a_vac,c_HfO2,edgeColor,lw)

F = faces6();

L=P.L; W=P.W; H=P.H;
H_Ti=P.H_Ti_bottom; H_AuB=P.H_gold_bottom;
L_vac=P.L_vac; H_vac=P.H_vac;
L_Au=P.L_Au_top; W_Au=P.W_Au_top; H_AuT=P.H_Au_top;
T=P.T_film;

% --- Bottom stack
V_base = vbox(o, L, W, H);
V_Ti   = vbox(o + [0 0 H], L, W, H_Ti);
V_AuB  = vbox(o + [0 0 H+H_Ti], L, W, H_AuB);

% --- Middle trench (Vacuum + HfO2 bodies)
z0_top = o(3) + H + H_Ti + H_AuB;
L_v = min(max(L_vac,0), L);
L_left  = max(0, (L - L_v)/2);
L_right = max(0, L - (L_left + L_v));

V_left  = vbox([o(1)                 o(2) z0_top], L_left,  W, H_vac);
V_vac   = vbox([o(1)+L_left          o(2) z0_top], L_v,     W, H_vac);
V_right = vbox([o(1)+L_left+L_v      o(2) z0_top], L_right, W, H_vac);

% --- Top Au cap (centered) + Ti rim
x0c = o(1) + (L - L_Au)/2;
y0c = o(2) + (W - W_Au)/2;
z0c = z0_top + H_vac;

V_cap   = vbox([x0c           y0c            z0c], L_Au,              W_Au,           H_AuT);
V_r_xL  = vbox([x0c - T       y0c            z0c], T,                 W_Au,           H_AuT);
V_r_xR  = vbox([x0c + L_Au    y0c            z0c], T,                 W_Au,           H_AuT);
V_r_yF  = vbox([x0c - T       y0c - T        z0c], L_Au + 2*T,        T,              H_AuT);
V_r_yB  = vbox([x0c - T       y0c + W_Au     z0c], L_Au + 2*T,        T,              H_AuT);

% --- External SiO2 fill (frame around the rim pack)
x_in_min = x0c - T;           x_in_max = x0c + L_Au + T;
y_in_min = y0c - T;           y_in_max = y0c + W_Au + T;
x_bmin   = o(1);              x_bmax   = o(1) + L;
y_bmin   = o(2);              y_bmax   = o(2) + W;

TxL = max(0, x_in_min - x_bmin);
TxR = max(0, x_bmax    - x_in_max);
TyF = max(0, y_in_min  - y_bmin);
TyB = max(0, y_bmax    - y_in_max);

V_fxL = []; V_fxR = []; V_fyF = []; V_fyB = [];
if TxL>0, V_fxL = vbox([x_bmin y_bmin z0c], TxL,           W,                 H_AuT); end
if TxR>0, V_fxR = vbox([x_in_max y_bmin z0c], TxR,         W,                 H_AuT); end
if TyF>0, V_fyF = vbox([x_in_min y_bmin z0c], x_in_max-x_in_min, TyF,         H_AuT); end
if TyB>0, V_fyB = vbox([x_in_min y_in_max z0c], x_in_max-x_in_min, TyB,       H_AuT); end

% --- Draw (legend from the first cell only)
drawPatch(ax,V_base, c_SiO2, edgeColor,lw, showLegend,'SiO2', 1, F);
drawPatch(ax,V_Ti,   c_Ti,   edgeColor,lw, showLegend,'Ti',   1, F);
drawPatch(ax,V_AuB,  c_Au,   edgeColor,lw, showLegend,'Au',   1, F);

drawPatch(ax,V_left, c_HfO2, edgeColor,lw, showLegend,'HfO2', 1, F);
drawPatch(ax,V_vac,  c_vac,  edgeColor,lw, showLegend,'vacuum', a_vac, F);
drawPatch(ax,V_right,c_HfO2, edgeColor,lw, false,'', 1, F, true);

drawPatch(ax,V_cap,   c_Au,  edgeColor,lw, false,'', 1, F, true);
drawPatch(ax,V_r_xL,  c_Ti,  edgeColor,lw, false,'', 1, F, true);
drawPatch(ax,V_r_xR,  c_Ti,  edgeColor,lw, false,'', 1, F, true);
drawPatch(ax,V_r_yF,  c_Ti,  edgeColor,lw, false,'', 1, F, true);
drawPatch(ax,V_r_yB,  c_Ti,  edgeColor,lw, false,'', 1, F, true);

drawPatch(ax,V_fxL,   c_SiO2,edgeColor,lw, false,'', 1, F, true);
drawPatch(ax,V_fxR,   c_SiO2,edgeColor,lw, false,'', 1, F, true);
drawPatch(ax,V_fyF,   c_SiO2,edgeColor,lw, false,'', 1, F, true);
drawPatch(ax,V_fyB,   c_SiO2,edgeColor,lw, false,'', 1, F, true);
end

% ---------- Filler: 4 stacked opaque layers ----------
function FL = buildFillerLayers(o, Lx, Wy, hvec)
% hvec = [h1 h2 h3 h4]
z = o(3);
FL.V1 = vbox([o(1) o(2) z], Lx, Wy, hvec(1)); z = z + hvec(1);
FL.V2 = vbox([o(1) o(2) z], Lx, Wy, hvec(2)); z = z + hvec(2);
FL.V3 = vbox([o(1) o(2) z], Lx, Wy, hvec(3)); z = z + hvec(3);
FL.V4 = vbox([o(1) o(2) z], Lx, Wy, hvec(4));
end

function drawFillerLayers(ax, FL, c1, c2, c3, c4, edgeCol, lw, F, legendName, offLegend)
if nargin<11, legendName=''; end
if nargin<12, offLegend=false; end

% layer 1 (optional legend)
if ~isempty(legendName) && ~offLegend
    patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1,'EdgeColor',edgeCol,'LineWidth',lw, ...
        'DisplayName',legendName);
else
    patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1,'EdgeColor',edgeCol,'LineWidth',lw, ...
        'HandleVisibility','off');
end
% layer 2
patch(ax,'Vertices',FL.V2,'Faces',F,'FaceColor',c2,'EdgeColor',edgeCol,'LineWidth',lw, ...
    'HandleVisibility','off');
% layer 3
patch(ax,'Vertices',FL.V3,'Faces',F,'FaceColor',c3,'EdgeColor',edgeCol,'LineWidth',lw, ...
    'HandleVisibility','off');
% layer 4
patch(ax,'Vertices',FL.V4,'Faces',F,'FaceColor',c4,'EdgeColor',edgeCol,'LineWidth',lw, ...
    'HandleVisibility','off');
end

% ======================================================================
% ============================== Helpers ===============================
% ======================================================================
function V = vbox(o,Lx,Wy,Hz)
% 8 vertices of an axis-aligned box (origin o, sizes Lx,Wy,Hz)
V = [ o;
      o + [Lx 0 0];
      o + [0 Wy 0];
      o + [0 0 Hz];
      o + [Lx Wy 0];
      o + [Lx 0 Hz];
      o + [0 Wy Hz];
      o + [Lx Wy Hz] ];
end

function F = faces6()
% 6 quad faces over the 8-vertex box
F = [1 2 5 3; 3 5 8 7; 1 3 7 4; 2 6 8 5; 1 4 6 2; 4 7 8 6];
end

function [c_SiO2,c_Ti,c_Au,c_vac,a_vac,c_HfO2,edgeColor,lw] = palette()
% Colors and styles
c_SiO2  = [0.55 0.35 0.20];
c_Ti    = [0.20 0.20 0.20];
c_Au    = [1.00 0.84 0.00];
c_vac   = [0.00 0.60 0.00];
a_vac   = 0.35;
c_HfO2  = [0.80 0.65 0.45];
edgeColor = 'k';
lw = 1.0;
end

function ph = drawPatch(axh,V,col,edgeCol,lw, addToLegend, name, faceAlpha, F, offLegend)
% Draw a patch with optional legend entry
if nargin<10, offLegend=false; end
if isempty(V), ph = []; return; end
args = {'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
        'EdgeColor',edgeCol,'LineWidth',lw};
if addToLegend && ~offLegend && ~isempty(name)
    ph = patch(axh, args{:}, 'DisplayName', name);
else
    ph = patch(axh, args{:}, 'HandleVisibility','off');
end
end
